<?php
/**
 * Dynamic CSS functions
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Get dynamic CSS
 */
function uncode_get_page_dynamic_css() {
	global $uncode_post_data, $uncode_dynamic_css_selectors, $row_sticky_columns;

	$uncode_dynamic_css_selectors = array();

	// Get an array that contains all the raw content attached to the page
	$content_array = uncode_get_post_data_content_array();

	// Generate custom CSS for the custom colors
	$dynamic_css = uncode_get_dynamic_colors_css( $content_array );

	// Generate custom CSS for the sticky scroll indexes
	$dynamic_css .= uncode_get_dynamic_sticky_scroll_css( $content_array );

	// Generate custom CSS for css grids
	$dynamic_css .= uncode_get_dynamic_css_grid_css( $content_array );

	return $dynamic_css;
}

/**
 * Get dynamic CSS generated by custom colors
 */
function uncode_get_dynamic_colors_css( $content_array ) {
	$css = '';

	foreach ( $content_array as $content ) {
		if ( strpos( $content, 'uncode-gradient' ) !== false || strpos( $content, 'uncode-solid' ) !== false ) {
			$css .= uncode_get_single_dynamic_colors_css( $content );
		}
	}

	return $css;
}

/**
 * Get dynamic colors CSS from a piece of content
 */
function uncode_get_single_dynamic_colors_css( $content ) {
	$css = '';

	$regex = '/\[([^\[]+) [^\s]+_type=\"(uncode-solid|uncode-gradient)\".*?\]/m';
	preg_match_all( $regex, $content, $shortcodes, PREG_SET_ORDER, 0 );

	$shortcodes_with_colors = uncode_get_shortcodes_with_colors( $shortcodes );

	foreach ( $shortcodes_with_colors as $shortcode ) {
		$shortcode_data = uncode_get_shortcodes_with_colors_data( $shortcode );

		if ( count( $shortcode_data ) > 0 ) {
			$css .= uncode_get_dynamic_colors_css_from_shortcode( $shortcode_data );
		}
	}

	return $css;
}

/**
 * Get shortcodes that have a custom color
 */
function uncode_get_shortcodes_with_colors( $shortcodes ) {
	$shortcodes_with_colors = array();

	foreach ( $shortcodes as $key => $value ) {
		if ( isset( $value[0] ) && $value[0] && isset( $value[2] ) && $value[2] ) {
			$shortcodes_with_colors[] = $value[0];
		}
	}

	return $shortcodes_with_colors;
}

/**
 * Get data of each shortcode that has a custom color
 */
function uncode_get_shortcodes_with_colors_data( $shortcode ) {
	$regex_id = '/\[(.+?\s+)/';
	preg_match_all( $regex_id, $shortcode, $shortcode_id, PREG_SET_ORDER, 0 );

	$shortcode_data = array(
		'type'       => '',
		'id'         => '',
		'attributes' => array(),
	);

	if ( isset( $shortcode_id[0] ) && $shortcode_id[0] && isset( $shortcode_id[0][1] ) && $shortcode_id[0][1] ) {
		$shortcode_id = trim( $shortcode_id[0][1] );

		if ( $shortcode_id ) {
			$regex_attr = '/ (.*?)=\"(.*?)\"/';
			preg_match_all( $regex_attr, $shortcode, $attributes, PREG_SET_ORDER, 0 );

			$shortcode_data['type'] = $shortcode_id;

			foreach ( $attributes as $attribute ) {
				if ( isset( $attribute[1] ) && $attribute[1] && isset( $attribute[2] ) && $attribute[2] ) {
					if ( $attribute[1] === 'uncode_shortcode_id' ) {
						$shortcode_data['id'] = absint( $attribute[2] );
					} else {
						$shortcode_data['attributes'][$attribute[1]] = $attribute[2];
					}
				}
			}
		}
	}

	if ( $shortcode_data['id'] > 0 ) {
		return $shortcode_data;
	}

	return array();
}

/**
 * Get CSS from custom color shortcode
 */
function uncode_get_dynamic_colors_css_from_shortcode( $shortcode ) {
	global $uncode_dynamic_css_selectors;

	$css = '';

	if ( isset( $shortcode['type'] ) && isset( $shortcode['id'] ) && isset( $shortcode['attributes'] ) && is_array( $shortcode['attributes'] ) ) {
		// Return early if we don't support this module
		if ( ! in_array( $shortcode['type'], uncode_get_modules_with_dynamic_colors() ) ) {
			return $css;
		}

		$shortcode_id = absint( $shortcode['id'] );

		// Check if we have already processed the same ID
		// (ie. a cloned module)
		if ( is_array( $uncode_dynamic_css_selectors ) && in_array( $shortcode_id, $uncode_dynamic_css_selectors ) ) {
			return $css;
		}

		$custom_color_keys = array();

		foreach ( $shortcode['attributes'] as $attributes_key => $attributes_value ) {
			if ( $attributes_value === 'uncode-solid' || $attributes_value === 'uncode-gradient' ) {
				$custom_color_keys[] = str_replace( '_type', '', $attributes_key );
			}
		}

		$shortcode_function_name = 'uncode_get_dynamic_colors_css_for_shortcode_' . str_replace( '-', '_', $shortcode['type'] );

		if ( function_exists( $shortcode_function_name ) ) {
			$css .= call_user_func_array( $shortcode_function_name, array( $shortcode, $custom_color_keys ) );
		}

		$uncode_dynamic_css_selectors[] = $shortcode['id'];
	}

	return $css;
}

/**
 * List of modules with dynamic colors
 */
function uncode_get_modules_with_dynamic_colors() {
	$modules = array(
		'uncode_author_profile',
		'vc_button',
		'vc_column',
		'vc_column_inner',
		'uncode_consent_notice',
		'vc_accordion',
		'vc_custom_heading',
		'vc_icon',
		'vc_gallery',
		'vc_message',
		'vc_row',
		'vc_row_inner',
		'vc_section',
		'vc_single_image',
		'vc_column_text',
		'vc_separator',
		'vc_gmaps',
		'vc_pie',
		'vc_tabs',
		'uncode_pricing',
		'uncode_list',
		'uncode_index',
		'uncode_counter',
		'uncode_vertical_text',
		'uncode_woocommerce_cart',
		'uncode_woocommerce_checkout',
		'uncode_woocommerce_my_account'
	);

	return $modules;
}

/**
 * Get dynamic CSS generated by sticky scroll
 */
function uncode_get_dynamic_sticky_scroll_css( $content_array ) {
	$menutype = ot_get_option('_uncode_headers');

	$css = '';

	foreach ( $content_array as $content ) {
		if ( strpos( $content, 'index_type="sticky-scroll"' ) !== false ) {
			$css .= uncode_get_single_dynamic_sticky_scroll_css( $content );
		}
	}

	return $css;
}

/**
 * Get dynamic sticky scroll CSS from a piece of content
 */
function uncode_get_single_dynamic_sticky_scroll_css( $content ) {
	$css = '';

	$regex_index_vw = '/\[[uncode_index|vc_gallery]([^\]]+)type=\"sticky-scroll\".*?\]/m';
	preg_match_all( $regex_index_vw, $content, $indexes_vw, PREG_SET_ORDER, 0 );

	foreach ( $indexes_vw as $index_vw ) {

		$params = $index_vw[0];
		preg_match('/ uncode_shortcode_id=\"(.*?)"/', $params, $id); //row ID
		preg_match('/ sticky_th_grid_lg=\"(.*?)"/', $params, $lg);
		preg_match('/ sticky_th_grid_md=\"(.*?)"/', $params, $md);
		preg_match('/ no_sticky_scroll_tablet=\"(.*?)"/', $params, $no_md);
		preg_match('/ sticky_th_grid_sm=\"(.*?)"/', $params, $sm);
		preg_match('/ no_sticky_scroll_mobile=\"(.*?)"/', $params, $no_sm);

		$shortcode_data = array(
			'size' => 'vw',
			'id' => $id[1],
			'lg' => isset($lg[1]) ? $lg[1] : 3,
			'md' => isset($no_md[1]) ? 1 : (isset($md[1]) ? $md[1] : 3),
			'sm' => isset($no_sm[1]) ? 1 : (isset($sm[1]) ? $sm[1] : 1),
		);
		$css .= uncode_get_dynamic_sticky_scroll_css_from_shortcode( $shortcode_data );

	}

	return $css;
}

/**
 * Get CSS from sticky scroll shortcode
 */
function uncode_get_dynamic_sticky_scroll_css_from_shortcode( $shortcode_data ) {
	global $uncode_dynamic_css_selectors, $row_sticky_columns;

	$css = '';

	if ( isset( $shortcode_data['id'] ) && isset( $shortcode_data['lg'] ) ) {

		$shortcode_id = absint( $shortcode_data['id'] );

		// Check if we have already processed the same ID
		// (ie. a cloned module)
		if ( is_array( $uncode_dynamic_css_selectors ) && in_array( 'hor-scroll-'.$shortcode_id, $uncode_dynamic_css_selectors ) ) {
			return $css;
		}

		if ( function_exists( 'uncode_get_sticky_scroll_css_for_shortcode_index' ) ) {
			$css .= uncode_get_sticky_scroll_css_for_shortcode_index( $shortcode_data );
		}

		$uncode_dynamic_css_selectors[] = 'hor-scroll-'.$shortcode_data['id'];
	}

	return $css;
}

/**
 * Get dynamic CSS generated by css grids
 */
function uncode_get_dynamic_css_grid_css( $content_array ) {
	$css = '';

	foreach ( $content_array as $content ) {
		if ( strpos( $content, 'type="css_grid"' ) !== false ) {
			$css .= uncode_get_single_dynamic_css_grid_css( $content );
		}
	}

	return $css;
}

/**
 * Get dynamic css grids CSS from a piece of content
 */
function uncode_get_single_dynamic_css_grid_css( $content ) {
	$css = '';

	$regex_index = '/\[[uncode_index|vc_gallery]([^\]]+)type=\"css_grid\".*?\]/m';
	preg_match_all( $regex_index, $content, $indexes, PREG_SET_ORDER, 0 );

	foreach ( $indexes as $index ) {
		$params = $index[0];

		preg_match('/ uncode_shortcode_id=\"(.*?)"/', $params, $id); //row ID
		preg_match('/ grid_items=\"(.*?)"/', $params, $items);
		preg_match('/ screen_lg_breakpoint=\"(.*?)"/', $params, $lg);
		preg_match('/ screen_lg_items=\"(.*?)"/', $params, $lg_items);
		preg_match('/ screen_md_breakpoint=\"(.*?)"/', $params, $md);
		preg_match('/ screen_md_items=\"(.*?)"/', $params, $md_items);
		preg_match('/ screen_sm_breakpoint=\"(.*?)"/', $params, $sm);
		preg_match('/ screen_sm_items=\"(.*?)"/', $params, $sm_items);

		$shortcode_data = array(
			'id'       => $id[1],
			'items'    => isset( $items[1] ) ? $items[1] : 4,
			'lg'       => isset( $lg[1] ) ? $lg[1] : 1000,
			'lg_items' => isset( $lg_items[1] ) ? $lg_items[1] : 3,
			'md'       => isset( $md[1] ) ? $md[1] : 600,
			'md_items' => isset( $md_items[1] ) ? $md_items[1] : 2,
			'sm'       => isset( $sm[1] ) ? $sm[1] : 480,
			'sm_items' => isset( $sm_items[1] ) ? $sm_items[1] : 1,
		);
		$css .= uncode_get_dynamic_css_grids_css_from_shortcode( $shortcode_data );
	}

	return $css;
}

/**
 * Get CSS from custom grids shortcode
 */
function uncode_get_dynamic_css_grids_css_from_shortcode( $shortcode_data ) {
	global $uncode_dynamic_css_selectors;

	$css = '';

	if ( isset( $shortcode_data['id'] ) && isset( $shortcode_data['lg'] ) ) {

		$shortcode_id = absint( $shortcode_data['id'] );

		// Check if we have already processed the same ID
		// (ie. a cloned module)
		if ( is_array( $uncode_dynamic_css_selectors ) && in_array( 'cssgrid-'.$shortcode_id, $uncode_dynamic_css_selectors ) ) {
			return $css;
		}

		if ( function_exists( 'uncode_get_css_grids_css_for_shortcode_index' ) ) {
			$css .= uncode_get_css_grids_css_for_shortcode_index( $shortcode_data );
		}

		$uncode_dynamic_css_selectors[] = 'cssgrid-'.$shortcode_data['id'];
	}

	return $css;
}
